(* astar.ml     A* heuristic search for Slide24
 *
 * Copyright (c) 2015 Psellos   http://psellos.com
 *
 * Licensed under the MIT license:
 *     http://www.opensource.org/licenses/mit-license.php
 *
 * This is a specialized version of the A* search algorithm:
 *     http://en.wikipedia.org/wiki/A*_search_algorithm
 *)
open Slide24defs

let config_match (a: config) (p: config) : bool =
    (* Return true iff the configuration a matches the pattern p, in
     * which nonnegative numbers match themselves and negative values
     * match anything.
     *)
    let length = Array.length a in
    let rec imatch k =
        if k >= length then true
        else if p.(k) >= 0 && a.(k) <> p.(k) then false
        else imatch (k + 1)
    in
    imatch 0


(* An inverse configuration has the same type as a configuration but
 * gives the inverse information; the kth entry I.(k) specifies where
 * the k tile is located.  Thus, I.(0) shows where the empty spot is.
 *)
type iconfig = config


let invert config : iconfig =
    (* Invert a configuration or a pattern, giving an inverse
     * configuration or an inverse pattern.
     *)
    let res = Array.make 25 (-1) in
    Array.iteri (fun i ci -> if ci >= 0 then res.(ci) <- i) config;
    res


let neighbors config : config list =
    (* Return the configurations reachable in one move from the given
     * configuration.
     *
     * (It's easiest to think of this as moving the empty spot in the 4
     * orthogonal directions.)
     *)
    let emptyx =
        let rec findit x =
            if config.(x) = 0 then x else findit (x + 1)
        in
        findit 0
    in
    let mtr = emptyx / 5 in
    let mtc = emptyx mod 5 in
    let add1 l (r, c) =
        if r >= 0 && r < 5 && c >= 0 && c < 5 then
            let config' = Array.copy config in
            let newx = r * 5 + c in
            let () = config'.(emptyx) <- config.(newx) in
            let () = config'.(newx) <- 0 in
            config' :: l
        else
            l
    in
    List.fold_left add1 []
        [(mtr, mtc + 1); (mtr, mtc - 1); (mtr + 1, mtc); (mtr - 1, mtc)]


let heuristic_pathlen (a: config) (pinv: iconfig) =
    (* Return the heuristic path length from configuration a to one that
     * matches inverse pattern pinv. In an inverse pattern, pinv.(t)
     * gives the cell where tile t should appear, or -1 if its location
     * is not constrained. Return a good guess for the number of tile
     * moves it would take to transform a to one matching pinv.
     *)
    let res = ref 0 in
    for r = 0 to 4 do
        for c = 0 to 4 do
            let tile = a.(r * 5 + c) in
            let ax = pinv.(tile) in
            if ax >= 0 then
                let dr = abs (ax / 5 - r) in
                let dc = abs (ax mod 5 - c) in
                let moves =
                    if tile = 0 then dr + dc
                    else if dc = 0 then
                        if dr = 0 then 0 else 5 * dr - 4
                    else
                        5 * dc - 4 + (if dr = 0 then 0 else 5 * dr - 2)
                in
                res := !res + moves
        done
    done;
    !res


(* Configurations traversed in the search.
 *)
module CMap =
    Map.Make (struct type t = config let compare = compare end)

(* Configs accessible by shortest heuristic path length to goal.
 *)
module CHeap =
    Set.Make (struct type t = int * config let compare = compare end)


(* State of a configuration on the frontier (waiting to be searched).
 *)
type cfstate =
    config  (* Predecessor = 'cfpred' *)
    * int   (* Path length from start to config = 'stplen' *)
    * int   (* Heuristic path length from config to goal = 'hgoplen' *)

(* State of the overall search.
 *)
type sstate =
    config CMap.t    (* visited config -> predecessor in best path *)
    * cfstate CMap.t (* frontier config -> state *)
    * CHeap.t        (* frontier configs by pathlen through them to goal *)

let sstate_cfstate sstate config : cfstate option =
    (* Get the state of a config on the frontier.
     *)
    let (_, cfsmap, _) = sstate in
    try Some (CMap.find config cfsmap)
    with Not_found -> None

let sstate_cfupdate sstate config ncfstate : sstate =
    (* Update the state of a config on the frontier.
     *)
    let (cpredmap, cfsmap, cheap) = sstate in
    let (ncfpred, nstplen, nhgoplen) = ncfstate in
    let cheap' =
        try
            let (_, ostplen, ohgoplen) = CMap.find config cfsmap in
            CHeap.remove (ostplen + ohgoplen, config) cheap
        with Not_found -> cheap
    in
    let cfsmap' = CMap.add config ncfstate cfsmap in
    (cpredmap, cfsmap', CHeap.add (nstplen + nhgoplen, config) cheap')


let sstate_shortest sstate : config * cfstate * sstate =
    (* Extract the configuration through which the goal can
     * heuristically be reached in the fewest steps. Return the
     * configuration, its pathlens, and the new overall search state.
     *)
    let (cpredmap, cfsmap, cheap) = sstate in
    let (_, config) as elt = CHeap.min_elt cheap in
    let cheap' = CHeap.remove elt cheap in
    let cfstate = CMap.find config cfsmap in
    let cfsmap' = CMap.remove config cfsmap in
    (config, cfstate, (cpredmap, cfsmap', cheap'))


let sstate_visit sstate config cpred =
    (* Update the search state to show that the given config has been
     * visited. Caller warrants that the config is no longer on the
     * frontier.
     *)
    let (cpredmap, cfsmap, cheap) = sstate in
    let cpredmap' = CMap.add config cpred cpredmap in
    (cpredmap', cfsmap, cheap)


let sstate_visited sstate config =
    (* Determine whether the config has been visited.
     *)
    let (cpredmap, _, _) = sstate in
    CMap.mem config cpredmap


let sstate_path sstate a b : config list =
    (* Figure out the path from a to b. Work backwards from b using the
     * predecessor info of sstate.
     *)
    let (cpredmap, _, _) = sstate in
    let rec lpath config sofar =
        if config = a then
            sofar
        else
            let pred = CMap.find config cpredmap in
            lpath pred (pred :: sofar)
    in
    lpath b [b]


let astar start goal maxexam : (config list * int) option =
    (* Return a list of configurations showing the moves required to get
     * from start to goal, where goal is a pattern. I.e., show how to
     * move tiles in start so that it matches goal.
     *
     * The search is limited by maxexam, the maximum number of
     * configurations to be examined. If the goal can't be reached
     * within this limit, return None. Otherwise return Some (configs,
     * examined), the results of the search and the number of
     * configurations examined along the way.
     *)
    let igoal = invert goal in

    let add_nabe config cstplen sstate nabe : sstate = 
        (* Add nabe, a neighbor of config, to the frontier of the search
         * state.
         *)
        if sstate_visited sstate nabe then
            sstate
        else
            match sstate_cfstate sstate nabe with
            | None ->
                let nhgoplen = heuristic_pathlen nabe igoal in
                sstate_cfupdate sstate nabe (config, cstplen + 1, nhgoplen)
            | Some (ncpred, nstplen, nhgoplen) ->
                if cstplen + 1 < nstplen then
                    sstate_cfupdate sstate nabe (config, cstplen + 1, nhgoplen)
                else
                    sstate
    in

    let rec dosearch sstate examined =
        if examined >= maxexam then
            None
        else
            let (config, (cpred, cstplen, _), sstate') = sstate_shortest sstate
            in
            let sstate'' = sstate_visit sstate' config cpred in
                if config_match config goal then
                    Some (sstate_path sstate'' start config, examined + 1)
                else
                    let nabes = neighbors config in
                    let sstate''' =
                        List.fold_left (add_nabe config cstplen) sstate'' nabes
                    in
                    dosearch sstate''' (examined + 1)
    in

    let cfstate0 = (start, 0, heuristic_pathlen start igoal) in
    let sstate0 = (CMap.empty, CMap.empty, CHeap.empty) in
    let sstate1 = sstate_cfupdate sstate0 start cfstate0 in
    dosearch sstate1 0
