(* slide24ViewControllerD.ml     View controller delegate for Slide24
 *
 * Copyright (c) 2015 Psellos   http://psellos.com
 *
 * Licensed under the MIT license:
 *     http://www.opensource.org/licenses/mit-license.php
 *)
open Cocoa
open Slide24defs

(* The app can be in five states
 *)
type state =
    | Normal      (* Under user control *)
    | Solving     (* Calculating solution: CPU intensive *)
    | Discouraged (* Under user control, but unable to solve *)
    | Inactive    (* In the background *)
    | Animating   (* Animating solution of puzzle, limited user control *)

(* Options for animation.
 *)
type anim =
    | AnimNone          (* No animation *)
    | AnimNorm          (* Animation with default duration *)
    | AnimDura of float (* Animation with specified duration (sec) *)


class t =
object (self)
    inherit Wrappee.t

    val mutable theView: UiView.t option = None
    val mutable vwd = 0.0 (* Width of view *)
    val mutable vht = 0.0 (* Height of view *)
    val mutable theShuffleB: UiButton.t option = None
    val mutable theSolveB: UiButton.t option = None

    val mutable tiles: UiButton.t array = [||]
    val mutable config: config = [||]
    val solved = Array.of_list (range 1 24 @ [0])
    val mutable solution: config list = []

    val mutable state: state = Normal

    (* Accessors.
     *)
    method private view =
        match theView with
        | None -> raise Not_found
        | Some v -> v

    method private setView' v =
        theView <- Some v

    method shuffle =
        match theShuffleB with
        | None -> raise Not_found
        | Some button -> button

    method setShuffle' button =
        theShuffleB <- Some button

    method solve =
        match theSolveB with
        | None -> raise Not_found
        | Some button -> button

    method setSolve' button =
        theSolveB <- Some button

    method launchWithView v =
        self#setView' v;
        let (_, _, w, h) = v#frame in
        vwd <- w;
        vht <- h;
        Random.self_init ();
        ignore (
            NsTimer.scheduledTimer5 interval self "timerTick'" Wrappee.nil true
        );
        self#make_tiles;
        let styleb bo =
            match bo with
            | None -> ()
            | Some button -> self#style_button button false;
        in
        styleb theShuffleB;
        styleb theSolveB

    method activate =
        state <- Normal;
        self#display

    method deactivate =
        state <- Inactive

    method viewController'viewWillAppear'
            (vc: Ui.slide24ViewController) (animated: bool) =
        ()

    method viewController'viewWillDisappear'
            (vc: Ui.slide24ViewController) (animated: bool) =
        ()

    method viewControllerShouldAutorotate' (vc: Ui.slide24ViewController) =
        false


    (* GUI events.
     *)
    method touchTile' (tile: UiButton.t) =
        (* Touched one of the tiles.
         *)
        match state with
        | Normal
        | Discouraged ->
            begin
            match self#rowcol_blocmove (self#rowcol_of_tile tile) with
            | (_, _, []) ->
                () (* Can't move the tile *)
            | (dr, dc, rcs) ->
                let dx, dy = float_of_int dc, float_of_int dr in
                UiView.beginAnimations'context' "touchTile" 0;
                ListLabels.iter rcs ~f:(
                    fun rc ->
                        let t = self#tile_of_rowcol rc
                        in let (x, y, w, h) = t#frame
                        in
                            t#setFrame' (x +. dx *. w, y +. dy *. h , w, h)
                );
                UiView.commitAnimations ();
                ListLabels.iter rcs ~f: (
                    fun (r, c) ->
                        let ix = r * 5 + c in
                        let ix' = ix + dr * 5 + dc in
                        config.(ix') <- config.(ix);
                        config.(ix) <- 0
                );
                state <- Normal;
                self#display
            end
        | _ -> ()

    method doShuffle' (b: UiButton.t) =
        match state with
        | Normal
        | Discouraged ->
            Slide24util.shuffle config 150;
            self#config_tiles (AnimDura 0.6);
            state <- Normal;
            self#display
        | _ -> ()

    method doSolve' (b: UiButton.t) =
        (match state with
        | Normal ->
            state <- Solving;
            (* Allow the GUI to update before we grab a chunk of CPU time.
             *)
            self#performSelector'withDelay' "runsolve" 0.3
        | Animating ->
            (* Stop solving.
             *)
            state <- Normal
        | _ -> ()
        );
        self#display

    method runsolve =
        (* This is rather CPU intensive, so we limit the search.  Every
         * now and then it will fail.
         *)
        begin
        (match Solve.solve config solved 20000 with
        | None ->
            state <- Discouraged
        | Some (soln, _) ->
            solution <- soln;
            state <- Animating
        );
        self#display;
        end

    method timerTick' (timer: NsTimer.t) =
        if state = Animating then
            match solution with
            | [] ->
                state <- Normal;
                self#display
            | config' :: rest ->
                config <- config';
                self#config_tiles AnimNorm;
                solution <- rest

    (* Private methods.
     *)
    method private make_tiles =
        (* Create tiles at launch.
         *)
        tiles <- ArrayLabels.init 24 ~f: (
            fun i ->
                let row = float_of_int (i / 5) in
                let col = float_of_int (i mod 5) in
                let tilenum = string_of_int (i + 1) in
                let b = UiButton.buttonWithType' UiButton.TypeCustom in
                let () =
                    begin
                    b#addTarget'action'forControlEvents'
                        self "touchTile'" [UiButton.EventTouchUpInside];
                    b#setFrame' (self#frame_of_tile row col);
                    b#setTitleColor'forState'
                        (if i mod 2 = 0 then red else white)
                        UiButton.kStateNormal;
                    b#setTitle'forState' tilenum UiButton.kStateNormal;
                    b#setBackgroundImage'forState'
                        ("tiles/tile" ^ tilenum ^ ".png")
                        UiButton.kStateNormal;
                    self#style_button b (i mod 2 = 0);
                    b#setTitleSize' 30.0;
                    match theView with
                    | None -> ()
                    | Some v -> v#addSubview' (b :> UiView.t)
                    end
                in
                b
        );
        config <- Array.copy solved

    method private style_button b is_white =
        begin
        b#layer#setCornerRadius' 7.0;
        b#layer#setMasksToBounds' true;
        b#layer#setBorderWidth' 2.0;
        b#layer#setBorderColor' (if is_white then dkwhite else dkred);
        end

    method private config_tiles anim =
        (* Move the tiles so they are in the positions given by the
         * current config.
         *)
        begin
        (match anim with
        | AnimNorm
        | AnimDura _ -> UiView.beginAnimations'context' "config_tiles" 0
        | _ -> ()
        );
        (match anim with
        | AnimDura d -> UiView.setAnimationDuration' d
        | _ -> ()
        );
        ArrayLabels.iteri config ~f: (
            fun i tilenum ->
                if tilenum <> 0 then
                    let row = float_of_int (i / 5) in
                    let col = float_of_int (i mod 5) in
                    let tile = tiles.(tilenum - 1) in
                    tile#setFrame' (self#frame_of_tile row col)
        );
        (match anim with
        | AnimNorm
        | AnimDura _ -> UiView.commitAnimations ()
        | _ -> ()
        );
        end

    method private frame_of_tile row col =
        let side = 60.0 in   (* Size of tiles (square) *)
        let left = frnd ((vwd -. 5.0 *. side) /. 2.0) in
        let top = frnd ((vht -. 5.0 *. side) /. 3.0) in
        (left +. side *. col, top +. side *. row, side, side)

    method private rowcol_of_number k : int * int =
        (* Find the row and column of the tile with the given number.
         * The number can also be 0, which returns the row and column of
         * the empty spot.
         *)
        let ix = array_findi ((=) k) config in
        (ix / 5, ix mod 5)

    method private rowcol_of_tile (tile: UiButton.t) : int * int =
        (* Return the row and column of the given tile.
         *)
        self#rowcol_of_number (int_of_string tile#currentTitle)

    method private tile_of_rowcol (row, col) : UiButton.t =
        tiles.(config.(row * 5 + col) - 1)

    method private rowcol_blocmove (tr, tc) : int * int * (int * int) list =
        (* If the tile at (tr, tc) can move (possibly en bloc with
         * others), return the unit (row, col) movement vector and the
         * (row, col) values of the tiles that move.  The first entry in
         * the list is the one closest to the empty spot.  If movement
         * is not possible, return a null vector and empty list.
         *)
        let null = (0, 0, []) in
        let vbloc drow col a b =
            (drow, 0, List.map (fun r -> (r, col)) (udrange a b))
        in
        let hbloc dcol row a b =
            (0, dcol, List.map (fun c -> (row, c)) (udrange a b))
        in
        let er, ec = self#rowcol_of_number 0 in
        if tr < er then
            if tc = ec then vbloc 1 tc (er - 1) tr else null
        else if tr > er then
            if tc = ec then vbloc (-1) tc (er + 1) tr else null
        else
            if tc < ec then
                hbloc 1 tr (ec - 1) tc
            else
                hbloc (-1) tr (ec + 1) tc

    method private display =
        (* Set the state of the buttons based on the state.
         *)
        match state with
        | Normal ->
            self#title_buttons "Shuffle" "Solve";
            self#enable_buttons true true
        | Solving ->
            self#title_buttons "Shuffle" "Working...";
            self#enable_buttons false false
        | Discouraged ->
            self#title_buttons "Shuffle" "Couldn't Solve";
            self#enable_buttons true false
        | Animating ->
            self#title_buttons "Shuffle" "Stop Solving";
            self#enable_buttons false true
        | _ -> ()

    method private title_buttons a b =
        let ltitle bo s =
            match bo with
            | None -> ()
            | Some button -> button#setTitle'forState' s UiButton.kStateNormal
        in
        ltitle theShuffleB a;
        ltitle theSolveB b

    method private enable_buttons a b =
        let lenable bo yesno =
            match bo with
            | None -> ()
            | Some button -> button#setEnabled' yesno
        in
        lenable theShuffleB a;
        lenable theSolveB b
end

let () =
    let wrapped robjcv =
        let c = new t in let () = c#setContainer robjcv in c
    in
    Callback.register "Slide24ViewControllerD.wrapped" wrapped
