(* slide24util.ml     Utility functions for Slide24 example
 *
 * Copyright (c) 2015 Psellos   http://psellos.com
 *
 * Licensed under the MIT license:
 *     http://www.opensource.org/licenses/mit-license.php
 *)
open Cocoa
open Slide24defs


let shuffle config n =
    (* Shuffle the board by moving the empty spot around randomly n
     * times.  It's an inefficient way to shuffle, but you can get
     * different amounts of shuffling by adjusting n.
     *)
    let empty_index =
        let rec findit x = if config.(x) = 0 then x else findit (x + 1) in
        findit 0
    in

    let rec ishuffle emptyx k =
        if k > 0 then
            let mtr = emptyx / 5 in
            let mtc = emptyx mod 5 in
            let (mtr', mtc') =
                if Random.bool () then
                    (mtr, mtc + Random.int 2 * 2 - 1)
                else
                    (mtr + Random.int 2 * 2 - 1, mtc)
            in
            if mtr' >= 0 && mtr' < 5 && mtc' >= 0 && mtc' < 5 then
                let emptyx' = mtr' * 5 + mtc' in
                let tmp = config.(emptyx') in
                begin
                config.(emptyx') <- 0;
                config.(emptyx) <- tmp;
                ishuffle emptyx' (k - 1);
                end
            else
                ishuffle emptyx k
    in

    ishuffle empty_index n


module CMap =
    Map.Make (struct type t = config let compare = compare end)

let streamline solution =
    (* Streamline the solution by removing redundant subsequences.  If a
     * configuration appears twice, the sequence between the two
     * appearances is redundant.  Repetitions show up fairly often at
     * the junctions between separately calculated sub-solutions, and
     * are somewhat tedious to watch.
     *)
    let check1 (seen, accum) config =
        if CMap.mem config seen then
            let accum' = CMap.find config seen in
            let rmct = List.length accum - List.length accum' in
            let seen' = List.fold_right CMap.remove (take rmct accum) seen in
            (seen', accum')
        else
            (CMap.add config (config :: accum) seen, config :: accum)
    in
    let (_, solution') = List.fold_left check1 (CMap.empty, []) solution in
    List.rev solution'
