(* solve.ml     Solve tile puzzle
 *
 * Copyright (c) 2015 Psellos   http://psellos.com
 *
 * Licensed under the MIT license:
 *     http://www.opensource.org/licenses/mit-license.php
 *)
open Slide24defs


let solverow current goal row maxexam : (config list * int) option =
    (* Solve one row of the tile puzzle by adding intermediate goals as
     * guidance. The given row should be less than 4, i.e., not the
     * bottom row. The given goal should be a configuration, not a
     * pattern.
     *
     * Using A* to solve the whole row often takes too long (too much
     * computation). Solving one tile at a time is much faster, but it
     * produces long move sequences (twice as long, say) and it also
     * gets hung up on the last tile of the row because it requires
     * moving other tiles out of the way; i.e., it requires moving
     * against the heuristic.
     *
     * To prevent the hangup, we create an intermediate goal where the
     * tiles are already moved out of the way:
     *
     *      2 3 4 5 .
     *      1 . . . .
     *
     * Then we switch to the real goal for the row, which A* can easily
     * figure out how to do.
     *
     * To keep move sequences reasonably short, we solve multiple tiles
     * at a time, but if this takes too long (too many configurations
     * examined) we drop back to solving fewer at a time.
     *)
    let lgoal = Array.make 25 0 in

    let rec solved k =
        (* Row already solved (for k = 5).
         *)
        k <= 0 ||
            let k' = k - 1 in
                current.(row * 5 + k') = goal.(row * 5 + k') && solved k'
    in

    let fill_midgoal k =
        Array.fill lgoal 0 25 (-1);
        Array.blit goal 0 lgoal 0 (row * 5);
        for i = 0 to min (k - 1) 3 do
            lgoal.(row * 5 + i) <- goal.(row * 5 + i + 1);
            if k > 4 then lgoal.(row * 5 + 5) <- goal.(row * 5);
        done
    in

    let fill_rowgoal () =
        Array.fill lgoal 0 25 (-1);
        Array.blit goal 0 lgoal 0 (row * 5 + 5)
    in

    let rec tryplace placed current examsofar trials =
        (* Attempt to make one step of progress by a series of smaller
         * and smaller numbers of tiles. Each trial has the form
         * (tilect, maxex), the number of new tiles to place, and how
         * many configs we're willing to examine to do it. Return None
         * on failure, or Some (mseq, tilect, examined), the new move
         * sequence, how many tiles were placed, and how many configs
         * were examined.
         *)
        match trials with
        | [] -> None
        | (tilect, maxex) :: rest ->
            let () = fill_midgoal (placed + tilect) in
            match Astar.astar current lgoal maxex with
            | None ->
                tryplace placed current (examsofar + maxex) rest
            | Some (mseq, examined) ->
                Some (mseq, tilect, examsofar + examined)
    in
    
    let rec midsolve placed mseq examsofar =
        (* Move tiles to the intermediate configuration described above.
         * Return None on failure or Some (mseq, examined).
         *)
        if placed = 5 then
            Some (mseq, examsofar)
        else
            let trials =
                if placed < 3 && examsofar < maxexam / 2 then
                    [(2, maxexam / 8); (1, maxexam - examsofar - maxexam / 8)]
                else
                    [(1, maxexam - examsofar)]
            in
            match tryplace placed (last mseq) examsofar trials with
            | None -> None
            | Some (mseqdelta, tilect, examined) ->
                let mseq' = mseq @ List.tl mseqdelta in
                midsolve (placed + tilect) mseq' examined
    in

    if solved 5 then
        Some ([current], 0)
    else
        match midsolve 0 [current] 0 with
        | None -> None
        | Some (mseq, examined) ->
            let current' = last mseq in
            let () = fill_rowgoal () in
            match Astar.astar current' lgoal (maxexam - examined) with
            | None -> None
            | Some (mseqdelta, examdelta) ->
                Some (mseq @ List.tl mseqdelta, examined + examdelta)


let solve start goal maxexam : (config list * int) option =
    (* Solve the tile puzzle; i.e., find a list of configurations that
     * show how to transform the start into the goal by sliding tiles.
     *
     * A direct solution by A* is frequently far too slow, so set
     * intermediate goals that help keep A* from getting hung up in
     * difficult spots. If successful, return a list of configurations
     * leading from start to goal, and the number of configurations that
     * were examined during the search.
     *
     * Maxexam gives the maximum number of configurations that should be
     * examined during the search. If this limit is exceeded before a
     * solution is found, return None.
     *)
    let solve1 (mseq, examsofar) row =
        if mseq = [] then
            (* Failure on a previous row.
             *)
            (mseq, examsofar)
        else
            let maxex = if row < 3 then maxexam / 4 else maxexam - examsofar in
            match solverow (last mseq) goal row maxex with
            | None -> ([], examsofar + maxex)
            | Some (mseqdelta, examdelta) ->
                (mseq @ List.tl mseqdelta, examsofar + examdelta)
    in
    match List.fold_left solve1 ([start], 0) [0; 1; 2; 3] with
    | ([], _) -> None
    | (configs, examct) ->
        match Astar.astar (last configs) goal (maxexam - examct) with
        | None -> None
        | Some (configs', examined) ->
            let rawsoln = configs @ List.tl configs' in
            Some (Slide24util.streamline rawsoln, examined)
