(* uiButton.ml     Wrapper for Cocoa Touch UIButton
 *
 * Copyright (c) 2015 Psellos   http://psellos.com
 *
 * Licensed under the MIT license:
 *     http://www.opensource.org/licenses/mit-license.php
 *)
open Cocoa

type buttonType =
    TypeCustom | TypeRoundedRect | TypeDetaileDisclosure | TypeInfoLight |
    TypeInfoDark | TypeContactAdd

(* A set of events is represented as a list.  We just define the touch
 * events.
 *)
type controlEvent = 
   EventTouchDown | EventTouchDownRepeat | EventTouchDragInside |
   EventTouchDragOutside | EventTouchDragEnter | EventTouchDragExit |
   EventTouchUpInside | EventTouchUpOutside | EventTouchCancel
type controlEvents = controlEvent list

(* What the API calls a state is closer to a set of states, again
 * represented as a list.
 *)
type controlState = StateHighlighted | StateDisabled | StateSelected
type controlStates = controlState list
let kStateNormal : controlStates = []

external _layer : nativeint -> CaLayer.t =
    "UIButton_layer"

external _frame : nativeint -> rect =
    "UIButton_frame"
external _setFrame_ : nativeint -> rect -> unit =
    "UIButton_setFrame_"

external _addTarget_action_forControlEvents_ :
    nativeint -> #Wrappee.t -> string -> controlEvents -> unit =
        "UIButton_addTarget_action_forControlEvents_"

external _enabled : nativeint -> bool =
    "UIButton_enabled"
external _setEnabled_ : nativeint -> bool -> unit =
    "UIButton_setEnabled_"

external _setTitle_forState_ : nativeint -> string -> controlStates -> unit =
    "UIButton_setTitle_forState_"
external _setTitleColor_forState_ :
    nativeint -> color -> controlStates -> unit =
        "UIButton_setTitleColor_forState_"
external _titleColorForState_ : nativeint -> controlStates -> color =
    "UIButton_titleColorForState_"
external _titleForState_ : nativeint -> controlStates -> string =
    "UIButton_titleForState_"
external _titleSize : nativeint -> float =
    "UIButton_titleSize"
external _setTitleSize_ : nativeint -> float -> unit =
    "UIButton_setTitleSize_"

external _setBackgroundImage_forState_ :
    nativeint -> image -> controlStates -> unit =
        "UIButton_setBackgroundImage_forState_"

external _currentTitle : nativeint -> string =
    "UIButton_currentTitle"
external _currentTitleColor : nativeint -> color =
    "UIButton_currentTitleColor"

class t robjcv =
    object (self)
    (* inherit Wrapper.t robjcv *)
    inherit UiView.t robjcv

    method layer = _layer self#contents

    method frame = _frame self#contents
    method setFrame' rect = _setFrame_ self#contents rect

    method addTarget'action'forControlEvents' :
                'a . (#Wrappee.t as 'a) -> string -> controlEvents -> unit =
        fun targ sel cevents ->
            _addTarget_action_forControlEvents_ self#contents targ sel cevents

    method enabled = _enabled self#contents
    method setEnabled' b = _setEnabled_ self#contents b

    method setTitle'forState' str cstates =
        _setTitle_forState_ self#contents str cstates
    method setTitleColor'forState' col cstates =
        _setTitleColor_forState_ self#contents col cstates
    method titleColorforState' cstates =
        _titleColorForState_ self#contents cstates
    method titleForState' cstates =
        _titleForState_ self#contents cstates
    method titleSize =
        _titleSize self#contents
    method setTitleSize' pts =
        _setTitleSize_ self#contents pts

    method setBackgroundImage'forState' img cstates =
        _setBackgroundImage_forState_ self#contents img cstates

    method currentTitle =
        _currentTitle self#contents
    method currentTitleColor =
        _currentTitleColor self#contents
    end

external buttonWithType' : buttonType -> t = "UIButton_buttonWithType_"

let () = Callback.register "UiButton.wrap" (new t)
