/* wrap.m     Custom wrappers for Slide24
 *
 * Copyright (c) 2015 Psellos   http://psellos.com
 *
 * Licensed under the MIT license:
 *     http://www.opensource.org/licenses/mit-license.php
 */
# import <stdlib.h>
# import <objc/runtime.h>
# import <Quartzcore/Quartzcore.h>
# import <UIKit/UIKit.h>
# import <caml/mlvalues.h>
# import <caml/memory.h>
# import <caml/alloc.h>
# import <caml/callback.h>
# import "wrap.h"

static value Val_fourtuple(double a, double b, double c, double d);
static void Fourtuple_val(double *ap, double *bp, double *cp, double *dp,
                        value tuple);
static unsigned Bitset_val(value bitset);
static value Val_UIColor(UIColor *color);
static UIColor *UIColor_val(value colorval);
static value Val_CGColor(CGColorRef color);
static CGColorRef CGColor_val(value colorval);
static SEL SEL_val(value selectval);


/* Represent raw ObjC object pointers in OCaml as nativeint values.
 */
static value wrapRawObjC(id obj)
{
    CAMLparam0();
    CAMLreturn(caml_copy_nativeint((intnat) obj));
}

static id unwrapRawObjC(value objcval)
{
    return (__bridge id) (void *) Nativeint_val(objcval);
}


/* Some key ObjC objects are wrapped up for access from OCaml. To keep
 * things simple, we enter wrapped ObjC objects into an NSMapTable and
 * never remove them. The table causes the objects to be persistent, and
 * it also lets us map from ObjC objects (pointers) to the OCaml objects
 * that wrap them (values).
 *
 * To stay on good terms with the OCaml GC, the OCaml values must be
 * reachable from a global root. For simplicity, we just make them all
 * global roots individually.
 * 
 * In this simple example these few ObjC objects are never destroyed.
 * In a more dynamic setting, we would need to gain control when the
 * OCaml object is GCed. Then we would remove the entry from the table,
 * which would cause (or allow) it to be deallocated on the ObjC side.
 */
static NSMapTable *g_objc_to_ocaml;

static void init_objc_to_ocaml()
{
    if (g_objc_to_ocaml)
        return;
    NSMapTableOptions kopt =
        NSMapTableStrongMemory | NSMapTableObjectPointerPersonality;
    NSMapTableOptions vopt =
        NSMapTableStrongMemory;
    g_objc_to_ocaml =
        [NSMapTable mapTableWithKeyOptions: kopt valueOptions: vopt];
}


static value find_ocaml_wrapper(id obj) {
    init_objc_to_ocaml();
    NSValue *v = [g_objc_to_ocaml objectForKey: obj];
    if (v == nil)
        return 0;
    return * (value *) [v pointerValue];
}


static value wrapObjC(char *wrapcs, id obj)
{
    /* Create a new OCaml object that wraps the given ObjC object.
     * wrapcs gives the name of the closure for creating an OCaml
     * wrapper of the correct type. A closure is registered by each
     * OCaml class that wraps an ObjC class. Each of these OCaml classes
     * inherits from Wrapper.t.
     */
    CAMLparam0();
    CAMLlocal1(val);
    value *valp;
    value *closure;

    closure = (value *) caml_named_value(wrapcs);
    valp = (value *) malloc(sizeof *valp);
    if(closure == NULL || valp == NULL) {
        /* If this happens, things will go very bad from here.
         */
        fprintf(stderr, "wrapObjC failure for %s\n", wrapcs);
        fflush(stderr);
        CAMLreturn(Val_unit); /* Not type safe, should never happen */
    }
    val = caml_callback(*closure, wrapRawObjC(obj));
    *valp = val;
    NSValue *v = [NSValue valueWithPointer: valp];

    /* Enter the wrapping into the table.
     */
    init_objc_to_ocaml();
    [g_objc_to_ocaml setObject: v forKey: obj];
    caml_register_global_root(valp);

    CAMLreturn(val);
}


static value wrapObjC_dyn(id obj)
{
    /* Wrap as wrapObjC(), but obtain the wrapping function name by
     * querying the class name of the object. Be willing to use the
     * wrapping function of a superclass if there is one.
     *
     * Note: this function isn't used right now, but it seems
     * potentially cool.
     */
    Class cl;
    const char *classname;
    char closurename[128];
#   define WRAP ".wrap"

    for (cl = object_getClass(obj); cl != nil; cl = class_getSuperclass(cl)) {
        classname = class_getName(cl);
        if(strlen(classname) + sizeof WRAP > sizeof closurename) {
            fprintf(stderr, "wrapObjC_dyn: class name too long: %s\n",
                    classname);
            fflush(stderr);
            return Val_unit;
        }
        strcpy(closurename, classname);
        strcat(closurename, WRAP);
        if (caml_named_value(closurename) != NULL)
            return wrapObjC(closurename, obj);
    }
    fprintf(stderr, "wrapObjC_dyn: lookup failure for %s\n", closurename);
    fflush(stderr);
    return Val_unit;
}


static id unwrapObjC(value obj) {
    /* Return the ObjC object wrapped by the given OCaml object.
     */
    CAMLparam1(obj);
    CAMLlocal1(objcval);

    objcval =
        caml_callback(
            caml_get_public_method(obj, caml_hash_variant("contents")),
            obj);
    CAMLreturnT(id, unwrapRawObjC(objcval));
}


/* Subclasses of WrapOCaml wrap OCaml objects as ObjC objects.
 */
@implementation WrapOCaml {
    value _contents;
}

- (value) contents
{
    return _contents;
}

- (void) setContents: (value) v
{
    _contents = v;
}

- (WrapOCaml *) init
{
    /* Create a new OCaml object that is wrapped by self. We match the
     * OCaml class to our own class by name--simple but effective. We
     * then call the closure for creating a wrapped object, which is
     * registered by each OCaml class that can be wrapped in an ObjC
     * class. Such OCaml classes inherit from Wrappee.t.
     */
    const char *classname;
    char closurename[128];
    value *closure;
#   define WRAPPED ".wrapped"

    if((self = [super init]) != nil) {
        classname = object_getClassName(self);
        if(strlen(classname) + sizeof WRAPPED > sizeof closurename) {
            fprintf(stderr, "[WrapOCaml init]: class name too long: %s\n",
                    classname);
            fflush(stderr);
            return nil;
        }
        strcpy(closurename, classname);
        strcat(closurename, WRAPPED);
        closure = (value *) caml_named_value(closurename);
        if(closure == NULL) {
            fprintf(stderr, "[WrapOCaml init]: lookup failure for %s\n",
                    closurename);
            fflush(stderr);
            return nil;
        }
        _contents = caml_callback(*closure, wrapRawObjC(self));
        caml_register_global_root(&_contents);
    }
    return self;
}

- (void) dealloc
{
    if(_contents != 0) {
        caml_remove_global_root(&_contents);
        _contents = 0;
    }
}

@end

value Wrappee_performSelector_withDelay_(value objcval, value selval,
                    value delayval)
/* nativeint -> string -> float -> unit */
{
    /* Any subclass of Wrappee.t can queue a selector to be delivered on
     * the main thread at a later time. If the delay is 0.0, it's a way
     * of yielding to the other parties in the thread (i.e., the GUI)
     * before continuing.
     */
    CAMLparam3(objcval, selval, delayval);

    NSObject *wrapper = (NSObject *) unwrapRawObjC(objcval);
    [wrapper performSelector: SEL_val(selval)
            withObject: nil
            afterDelay: Double_val(delayval)];

    CAMLreturn(Val_unit);
}


/* OCaml objects accessed from ObjC.
 */
@implementation Slide24AppDelegate

@dynamic window;

- (UIWindow *) window
{
    CAMLparam0();
    CAMLlocal2(selfval, windowval);

    selfval = [self contents];
    windowval =
        caml_callback(
            caml_get_public_method(selfval, caml_hash_variant("window")),
            selfval);
    CAMLreturnT(UIWindow *, unwrapObjC(windowval));
}

- (void) setWindow: (UIWindow *) aWindow
{
    CAMLparam0();
    CAMLlocal2(selfval, windowval);

    selfval = [self contents];
    windowval = wrapObjC("UiWindow.wrap", (id) aWindow);
    (void)
        caml_callback2(
            caml_get_public_method(selfval, caml_hash_variant("setWindow'")),
            selfval, windowval);
    CAMLreturn0;
}

- (void) applicationDidFinishLaunching: (UIApplication *) anApplication
{
    CAMLparam0();
    CAMLlocal2(selfval, applval);

    selfval = [self contents];
    if((applval = find_ocaml_wrapper(anApplication)) == 0)
        applval = wrapObjC("UiApplication.wrap", (id) anApplication);
    (void)
        caml_callback2(
            caml_get_public_method(
                selfval,
                caml_hash_variant("applicationDidFinishLaunching'")),
            selfval, applval);
    CAMLreturn0;
}

- (void) applicationWillResignActive: (UIApplication *) anApplication
{
    CAMLparam0();
    CAMLlocal2(selfval, applval);

    selfval = [self contents];
    if((applval = find_ocaml_wrapper(anApplication)) == 0)
        applval = wrapObjC("UiApplication.wrap", (id) anApplication);
    (void)
        caml_callback2(
            caml_get_public_method(
                selfval,
                caml_hash_variant("applicationWillResignActive'")),
            selfval, applval);
    CAMLreturn0;
}

- (void) applicationDidBecomeActive: (UIApplication *) anApplication
{
    CAMLparam0();
    CAMLlocal2(selfval, applval);

    selfval = [self contents];
    if((applval = find_ocaml_wrapper(anApplication)) == 0)
        applval = wrapObjC("UiApplication.wrap", (id) anApplication);
    (void)
        caml_callback2(
            caml_get_public_method(
                selfval,
                caml_hash_variant("applicationDidBecomeActive'")),
            selfval, applval);
    CAMLreturn0;
}
@end

@implementation Slide24ViewControllerD : WrapOCaml

@dynamic shuffle;

- (UIButton *) shuffle
{
    CAMLparam0();
    CAMLlocal2(selfval, buttonval);

    selfval = [self contents];
    buttonval =
        caml_callback(
            caml_get_public_method(selfval, caml_hash_variant("shuffle")),
            selfval);
    CAMLreturnT(UIButton *, unwrapObjC(buttonval));
}

- (void) setShuffle: (UIButton *) aButton
{
    CAMLparam0();
    CAMLlocal2(selfval, buttonval);

    selfval = [self contents];
    buttonval = wrapObjC("UiButton.wrap", (id) aButton);
    (void)
        caml_callback2(
            caml_get_public_method(selfval, caml_hash_variant("setShuffle'")),
            selfval,
            buttonval);
    CAMLreturn0;
}

@dynamic solve;

- (UIButton *) solve
{
    CAMLparam0();
    CAMLlocal2(selfval, buttonval);

    selfval = [self contents];
    buttonval =
        caml_callback(
            caml_get_public_method(selfval, caml_hash_variant("solve")),
            selfval);
    CAMLreturnT(UIButton *, unwrapObjC(buttonval));
}

- (void) setSolve: (UIButton *) aButton
{
    CAMLparam0();
    CAMLlocal2(selfval, buttonval);

    selfval = [self contents];
    buttonval = wrapObjC("UiButton.wrap", (id) aButton);
    (void)
        caml_callback2(
            caml_get_public_method(selfval, caml_hash_variant("setSolve'")),
            selfval,
            buttonval);
    CAMLreturn0;
}

- (void) touchTile: (UIButton *) aButton
{
    CAMLparam0();
    CAMLlocal2(selfval, buttonval);

    selfval = [self contents];
    if((buttonval = find_ocaml_wrapper(aButton)) == 0)
        /* Can't touch unknown button.
         */
        return;
    (void)
        caml_callback2(
            caml_get_public_method(selfval, caml_hash_variant("touchTile'")),
            selfval,
            buttonval);
    CAMLreturn0;
}

- (IBAction) doShuffle: (UIButton *) button
{
    CAMLparam0();
    CAMLlocal2(selfval, buttonval);

    selfval = [self contents];
    if((buttonval = find_ocaml_wrapper(button)) == 0)
        /* Can't touch unknown button.
         */
        return;
    (void)
        caml_callback2(
            caml_get_public_method(selfval, caml_hash_variant("doShuffle'")),
            selfval, buttonval);
    CAMLreturn0;
}

- (IBAction) doSolve: (UIButton *) button
{
    CAMLparam0();
    CAMLlocal2(selfval, buttonval);

    selfval = [self contents];
    if((buttonval = find_ocaml_wrapper(button)) == 0)
        /* Can't touch unknown button.
         */
        return;
    (void)
        caml_callback2(
            caml_get_public_method(selfval, caml_hash_variant("doSolve'")),
            selfval, buttonval);
    CAMLreturn0;
}

- (void) runsolve
{
    CAMLparam0();
    CAMLlocal1(selfval);

    selfval = [self contents];
    (void)
        caml_callback(
            caml_get_public_method(selfval, caml_hash_variant("runsolve")),
            selfval);

    CAMLreturn0;
}

- (void) timerTick: (NSTimer *) aTimer
{
    CAMLparam0();
    CAMLlocal2(selfval, timerval);

    selfval = [self contents];
    if((timerval = find_ocaml_wrapper(aTimer)) == 0)
        /* Can't report a timer tick if the timer isn't known.
         */
        return;
    (void)
        caml_callback2(
            caml_get_public_method(selfval, caml_hash_variant("timerTick'")),
            selfval, timerval);
    CAMLreturn0;
}

- (void) viewController: (UIViewController *) vc
         viewWillAppear: (BOOL) animated
{
    CAMLparam0();
    CAMLlocal2(selfval, vcval);

    selfval = [self contents];
    if ((vcval = find_ocaml_wrapper(vc)) == 0)
        vcval = wrapObjC("Slide24ViewController.wrap", (id) vc);
    (void)
        caml_callback3(
            caml_get_public_method(
                selfval,
                caml_hash_variant("viewController'viewWillAppear'")),
            selfval, vcval, Val_bool(!!animated));
    CAMLreturn0;
}

- (void) viewController: (UIViewController *) vc
         viewWillDisappear: (BOOL) animated
{
    CAMLparam0();
    CAMLlocal2(selfval, vcval);

    selfval = [self contents];
    if ((vcval = find_ocaml_wrapper(vc)) == 0)
        vcval = wrapObjC("PortlandViewController.wrap", (id) vc);
    (void)
        caml_callback3(
            caml_get_public_method(
                selfval,
                caml_hash_variant("viewController'viewWillDisappear'")),
            selfval, vcval, Val_bool(!!animated));
    CAMLreturn0;
}


- (BOOL) viewControllerShouldAutorotate: (UIViewController *) vc
{
    CAMLparam0();
    CAMLlocal3(selfval, vcval, resval);

    selfval = [self contents];
    if ((vcval = find_ocaml_wrapper(vc)) == 0)
        vcval = wrapObjC("PortlandViewController.wrap", (id) vc);
    resval =
        caml_callback2(
            caml_get_public_method(
                selfval,
                caml_hash_variant("viewControllerShouldAutorotate'")),
            selfval, vcval);
    CAMLreturnT(BOOL, Bool_val(resval));
}

@end


@implementation Slide24ViewController
/* (Just a thin wrapper that passes methods along to a delegate.)
 *  */

- (void) viewWillAppear: (BOOL) animated
{
    [self.delegate viewController: self viewWillAppear: animated];
}

- (void) viewWillDisappear: (BOOL) animated
{
    [self.delegate viewController: self viewWillDisappear: animated];
}

- (BOOL) shouldAutorotate
{
    return [self.delegate viewControllerShouldAutorotate: self];
}

@end

value Slide24ViewController_delegate(value objcval)
/* nativeint -> Slide24ViewControllerD.t */
{
    CAMLparam1(objcval);

    Slide24ViewController *svc = unwrapRawObjC(objcval);
    Slide24ViewControllerD *delg =  (Slide24ViewControllerD *) [svc delegate];
    CAMLreturn([delg contents]);
}

value Slide24ViewController_setDelegate_(value objcval, value delgval)
/* nativeint -> Slide24ViewControllerD.t -> unit */
{
    CAMLparam2(objcval, delgval);  // delgval : Slide24ViewControllerD.t
    CAMLlocal1(delgro);            // delgro : "raw" nativeint
    Slide24ViewControllerD *delg;  // delg: ObjC object

    Slide24ViewController *svc = unwrapRawObjC(objcval);
    delgro =
        caml_callback(
            caml_get_public_method(delgval, caml_hash_variant("container")),
            delgval);
    delg = unwrapRawObjC(delgro);
    [svc setDelegate: delg];
    CAMLreturn(Val_unit);
}

value Slide24ViewController_view(value objcval)
/* nativeint -> UiView.t */
{
    CAMLparam1(objcval);
    CAMLlocal1(vob);

    Slide24ViewController *svc = unwrapRawObjC(objcval);
    UIView *v = (UIView *) [svc view];
    vob = wrapObjC("UiView.wrap", v);
    CAMLreturn(vob);
}



/* ObjC objects accessed from OCaml.
 */

/* NSTimer
 */

value NSTimer_scheduledTimer5(value timeInterval, value target,
                    value selectval, value userInfo, value repeats)
/* float -> #Wrappee.t -> string -> #Wrappee.t -> bool -> NsTimer.t */
{
    /* For simplicity, the target must be wrapped (by an ObjC object)
     * already.  It's OK if userInfo isn't wrapped; we treat this as
     * nil.
     */
    CAMLparam5(timeInterval, target, selectval, userInfo, repeats);
    CAMLlocal2(targro, uiro);

    double ti = Double_val(timeInterval);
    targro = 
        caml_callback(
            caml_get_public_method(target, caml_hash_variant("container")),
            target);
    id targ = unwrapRawObjC(targro);
    if(targ == nil) {
        /* If the target wasn't already wrapped, things will go very bad
         * from here.
         */
        fprintf(stderr, "NSTimer_scheduledTimer5: unwrapped target\n");
        fflush(stderr);
        CAMLreturn(Val_unit); /* Not type safe, should never happen */
    }
    SEL sel = SEL_val(selectval);
    uiro =
        caml_callback(
            caml_get_public_method(userInfo, caml_hash_variant("container")),
            userInfo);
    id ui = unwrapRawObjC(uiro);
    int rep = Bool_val(repeats);

    NSTimer *timer =
        [NSTimer scheduledTimerWithTimeInterval: ti
                 target: targ
                 selector: sel
                 userInfo: ui
                 repeats: rep ];
    CAMLreturn(wrapObjC("NsTimer.wrap", (id) timer));
}


/* CALayer
 */

value CALayer_masksToBounds(value objcval)
/* nativeint -> bool */
{
    CAMLparam1(objcval);

    CALayer *layer = unwrapRawObjC(objcval);

    CAMLreturn(Val_bool([layer masksToBounds]));
}

value CALayer_setMasksToBounds_(value objcval, value boolval)
/* nativeint -> bool -> unit */
{
    CAMLparam2(objcval, boolval);

    CALayer *layer = unwrapRawObjC(objcval);
    [layer setMasksToBounds: Int_val(boolval)];

    CAMLreturn(Val_unit);
}

value CALayer_cornerRadius(value objcval)
/* nativeint -> float */
{
    CAMLparam1(objcval);

    CALayer *layer = unwrapRawObjC(objcval);

    CAMLreturn(caml_copy_double([layer cornerRadius]));
}

value CALayer_setCornerRadius_(value objcval, value floatval)
/* nativeint -> float -> unit */
{
    CAMLparam2(objcval, floatval);

    CALayer *layer = unwrapRawObjC(objcval);
    [layer setCornerRadius: Double_val(floatval)];

    CAMLreturn(Val_unit);
}

value CALayer_borderWidth(value objcval)
/* nativeint -> float */
{
    CAMLparam1(objcval);

    CALayer *layer = unwrapRawObjC(objcval);

    CAMLreturn(caml_copy_double([layer borderWidth]));
}

value CALayer_setBorderWidth_(value objcval, value floatval)
/* nativeint -> float -> unit */
{
    CAMLparam2(objcval, floatval);

    CALayer *layer = unwrapRawObjC(objcval);
    [layer setBorderWidth: Double_val(floatval)];

    CAMLreturn(Val_unit);
}

value CALayer_borderColor(value objcval)
/* nativeint -> color */
{
    CAMLparam1(objcval);

    CALayer *layer = unwrapRawObjC(objcval);

    CAMLreturn(Val_CGColor([layer borderColor]));
}

value CALayer_setBorderColor_(value objcval, value colorval)
/* nativeint -> color -> unit */
{
    CAMLparam2(objcval, colorval);

    CALayer *layer = unwrapRawObjC(objcval);
    CGColorRef cgcolor = CGColor_val(colorval);
    [layer setBorderColor: cgcolor];

    CAMLreturn(Val_unit);
}

/* UIView
 */

value UIView_frame(value objcval)
/* nativeint -> Cocoa.rect */
{
    CAMLparam1(objcval);
    CAMLlocal1(rectval);
    CGRect rect;

    UIView *view = unwrapRawObjC(objcval);
    rect = [view frame];
    rectval = Val_fourtuple(rect.origin.x, rect.origin.y,
                            rect.size.width, rect.size.height);
    CAMLreturn(rectval);
}

value UIView_addSubview_(value objcval, value subvval)
/* nativeint -> nativeint -> unit */
{
    CAMLparam2(objcval, subvval);

    UIView *view = unwrapRawObjC(objcval);
    UIView *subview = unwrapRawObjC(subvval);
    [view addSubview: subview];

    CAMLreturn(Val_unit);
}

value UIView_beginAnimations_context_(value animidval, value ctxval)
/* string -> int -> unit */
{
    CAMLparam2(animidval, ctxval);

    NSString *animid = [NSString stringWithUTF8String: String_val(animidval)];
    [UIView beginAnimations: animid context: (void *) Int_val(ctxval)];

    CAMLreturn(Val_unit);
}

value UIView_commitAnimations(value unitval)
/* unit -> unit */
{
    CAMLparam1(unitval);
    [UIView commitAnimations];
    CAMLreturn(Val_unit);
}

value UIView_setAnimationDuration_(value duraval)
/* float -> unit */
{
    CAMLparam1(duraval);
    [UIView setAnimationDuration: Double_val(duraval)];
    CAMLreturn(Val_unit);
}

/* UIWindow
 */

value UIWindow_rootViewController(value objcval)
/* nativeint -> Ui.slide24ViewController */
{
    /* (We assume root view controller is a Slide24ViewController.)
     */
    CAMLparam1(objcval);
    CAMLlocal1(vcob);

    UIWindow *window = unwrapRawObjC(objcval);
    Slide24ViewController *vc =
        (Slide24ViewController *) [window rootViewController];
    vcob = wrapObjC("Slide24ViewController.wrap", vc);
    CAMLreturn(vcob);
}



/* UIButton
 */

value UIButton_buttonWithType_(value btyval)
/* buttonType -> UiButton.t */
{
    CAMLparam1(btyval);

    UIButton *button = [UIButton buttonWithType: Int_val(btyval)];

    CAMLreturn(wrapObjC("UiButton.wrap", (id) button));
}

value UIButton_layer(value objcval)
/* nativeint -> CALayer.t */
{
    CAMLparam1(objcval);
    CAMLlocal1(layerval);
    CALayer *layer;

    UIButton *button = unwrapRawObjC(objcval);
    layer = [button layer];

    if((layerval = find_ocaml_wrapper(layer)) == 0)
        layerval = wrapObjC("CaLayer.wrap", (id) layer);

    CAMLreturn(layerval);
}

value UIButton_frame(value objcval)
/* nativeint -> rect */
{
    CAMLparam1(objcval);
    CAMLlocal1(res);

    UIButton *button = unwrapRawObjC(objcval);
    CGRect frame = [button frame];
    res =
        Val_fourtuple(frame.origin.x, frame.origin.y,
                        frame.size.width, frame.size.height);
    CAMLreturn(res);
}

value UIButton_addTarget_action_forControlEvents_(value objcval, value targval,
                    value actval, value evval)
/* nativeint -> #Wrappee.t -> string -> controlEvents -> unit */
{
    CAMLparam4(objcval, targval, actval, evval);
    CAMLlocal1(targro);
    char *prime;

    UIButton *button = unwrapRawObjC(objcval);
    targro =
        caml_callback(
            caml_get_public_method(targval, caml_hash_variant("container")),
            targval);
    id targ = unwrapRawObjC(targro);
    [button addTarget: targ
            action: SEL_val(actval)
            forControlEvents: Bitset_val(evval)];

    CAMLreturn(Val_unit);
}

value UIButton_enabled(value objcval)
/* nativeint -> bool */
{
    CAMLparam1(objcval);

    UIButton *button = unwrapRawObjC(objcval);

    CAMLreturn(Val_bool([button isEnabled]));
}

value UIButton_setEnabled_(value objcval, value boolval)
/* nativeint -> bool -> unit */
{
    CAMLparam2(objcval, boolval);

    UIButton *button = unwrapRawObjC(objcval);
    [button setEnabled: Bool_val(boolval)];

    CAMLreturn(Val_unit);
}

value UIButton_setFrame_(value objcval, value rectval)
/* nativeint -> rect -> unit */
{
    CAMLparam2(objcval, rectval);
    CGRect frame;

    UIButton *button = unwrapRawObjC(objcval);
    frame.origin.x = Double_val(Field(rectval, 0));
    frame.origin.y = Double_val(Field(rectval, 1));
    frame.size.width = Double_val(Field(rectval, 2));
    frame.size.height = Double_val(Field(rectval, 3));

    [button setFrame: frame];
    CAMLreturn(Val_unit);
}

value UIButton_setTitle_forState_(value objcval, value titleval, value stval)
/* nativeint -> string -> controlStates -> unit */
{
    CAMLparam3(objcval, titleval, stval);

    UIButton *button = unwrapRawObjC(objcval);
    NSString *title = [NSString stringWithUTF8String: String_val(titleval)];

    [button setTitle: title forState: Bitset_val(stval)];
    CAMLreturn(Val_unit);
}

value UIButton_setTitleColor_forState_(value objcval, value colorval,
                        value stval)
/* nativeint -> color -> controlStates -> unit */
{
    CAMLparam3(objcval, colorval, stval);

    UIButton *button = unwrapRawObjC(objcval);
    UIColor *color = UIColor_val(colorval);

    [button setTitleColor: color forState: Bitset_val(stval)];
    CAMLreturn(Val_unit);
}

value UIButton_titleColorForState_(value objcval, value stval)
/* nativeint -> controlStates -> color */
{
    CAMLparam2(objcval, stval);

    UIButton *button = unwrapRawObjC(objcval);
    UIColor *color = [button titleColorForState: Bitset_val(stval)];

    CAMLreturn(Val_UIColor(color));
}

value UIButton_titleForState_(value objcval, value stval)
/* nativeint -> controlStates -> string */
{
    CAMLparam2(objcval, stval);

    UIButton *button = unwrapRawObjC(objcval);
    NSString *title = [button titleForState: Bitset_val(stval)];

    CAMLreturn(caml_copy_string([title UTF8String]));
}

value UIButton_titleSize(value objcval)
/* nativeint -> float */
{
    /* This is a synthetic method.  In a richer setting you might
     * provide access to the title label [UIButton titleLabel] and wrap
     * the UILabel and UIFont classes.
     */
    CAMLparam1(objcval);

    UIButton *button = unwrapRawObjC(objcval);
    CGFloat size = [[[button titleLabel] font] pointSize];

    CAMLreturn(caml_copy_double((double) size));
}

value UIButton_setTitleSize_(value objcval, value ptsval)
/* nativeint -> float -> unit */
{
    /* Another synthetic method like the previous.
     */
    CAMLparam2(objcval, ptsval);

    UIButton *button = unwrapRawObjC(objcval);
    UIFont *font = [UIFont systemFontOfSize: Double_val(ptsval)];
    [[button titleLabel] setFont: font];

    CAMLreturn(Val_unit);
}

value UIButton_setBackgroundImage_forState_(value objcval, value imageval,
                    value stval)
/* nativeint -> image -> controlStates -> unit */
{
    CAMLparam3(objcval, imageval, stval);

    UIButton *button = unwrapRawObjC(objcval);
    NSString *name = [NSString stringWithUTF8String: String_val(imageval)];
    UIImage *image = [UIImage imageNamed: name];
    [button setBackgroundImage: image forState: Bitset_val(stval)];

    CAMLreturn(Val_unit);
}

value UIButton_currentTitle(value objcval)
/* nativeint -> string */
{
    CAMLparam1(objcval);

    UIButton *button = unwrapRawObjC(objcval);
    NSString *title = [button currentTitle];

    CAMLreturn(caml_copy_string([title UTF8String]));
}

value UIButton_currentTitleColor(value objcval)
/* nativeint -> color */
{
    CAMLparam1(objcval);

    UIButton *button = unwrapRawObjC(objcval);
    UIColor *color = [button currentTitleColor];

    CAMLreturn(Val_UIColor(color));
}


/* Utilities
 */
static value Val_fourtuple(double a, double b, double c, double d)
{
    CAMLparam0();
    CAMLlocal1(res);

    res = caml_alloc_tuple(4);
    Store_field(res, 0, caml_copy_double(a));
    Store_field(res, 1, caml_copy_double(b));
    Store_field(res, 2, caml_copy_double(c));
    Store_field(res, 3, caml_copy_double(d));

    CAMLreturn(res);
}


static void Fourtuple_val(double *ap, double *bp, double *cp, double *dp,
                        value tuple)
{
    CAMLparam1(tuple);

    *ap = Double_val(Field(tuple, 0));
    *bp = Double_val(Field(tuple, 1));
    *cp = Double_val(Field(tuple, 2));
    *dp = Double_val(Field(tuple, 3));

    CAMLreturn0;
}


static unsigned Bitset_val(value bitsetval)
{
    CAMLparam1(bitsetval);
    unsigned res;

    res = 0;
    while(bitsetval != Val_emptylist) {
        res |= 1 << Int_val(Field(bitsetval, 0));   /* car */
        bitsetval = Field(bitsetval, 1);            /* cdr */
    }

    CAMLreturnT(unsigned, res);
}


static value Val_UIColor(UIColor *color)
{
    return Val_CGColor([color CGColor]);
}


static UIColor *UIColor_val(value colorval)
{
    CAMLparam1(colorval);
    double a, b, c, d;
    UIColor *res;

    Fourtuple_val(&a, &b, &c, &d, colorval);
    res = [UIColor colorWithRed: a green: b blue: c alpha: d];

    CAMLreturnT(UIColor *, res);
}


static value Val_CGColor(CGColorRef cgcolor)
{
    CAMLparam0();
    double rgba[4];
    int ct, i;

    /* Colors will always be RGBA, but it doesn't hurt to be careful.
     */
    for(i = 0; i < 3; i++)
        rgba[i] = 0.0;
    rgba[3] = 1.0;

    if((ct = CGColorGetNumberOfComponents(cgcolor)) > 4)
        ct = 4;

    const CGFloat *comps = CGColorGetComponents(cgcolor);
    for(i = 0; i < ct; i++)
        rgba[i] = comps[i];

    CAMLreturn(Val_fourtuple(rgba[0], rgba[1], rgba[2], rgba[3]));
}


static CGColorRef CGColor_val(value colorval)
{
    CAMLparam1(colorval);

    /* (Defer colorspace handling to UIColor.)
     */
    UIColor *color = UIColor_val(colorval);

    CAMLreturnT(CGColorRef, [color CGColor]);
}


static SEL SEL_val(value selectval)
{
    /* Translate the OCaml string into an ObjC selector.
     *
     * Conventionally we use '\'' in OCaml where ':' appears in ObjC.
     * So here we translate back.
     */
    char *prime;
    SEL res;

    char *ocamlsel = String_val(selectval);
    char *buf = malloc(strlen(ocamlsel) + 1);
    strcpy(buf, ocamlsel);
    prime = buf;
    while((prime = strchr(prime, '\'')) != NULL)
        *prime = ':';
    res = sel_registerName(buf);
    free(buf);
    return res;
}
